﻿namespace Hims.Api.Controllers
{
    using System;
    using System.Threading.Tasks;
    using Domain.Helpers;
    using Domain.Services;
    using Microsoft.AspNetCore.Authorization;
    using Microsoft.AspNetCore.Mvc;
    using Models.Account;
    using Shared.DataFilters;
    using Shared.EntityModels;
    using Shared.Library.Enums;

    /// <summary>
    /// The password controller.
    /// </summary>
    public class PasswordController : Controller
    {
        /// <summary>
        /// The account services.
        /// </summary>
        private readonly IAccountService accountServices;

        /// <summary>
        /// The account credential services.
        /// </summary>
        private readonly IAccountCredentialService accountCredentialServices;

        /// <summary>
        /// The audit log services.
        /// </summary>
        private readonly IAuditLogService auditLogServices;

        /// <summary>
        /// The AES helper.
        /// </summary>
        private readonly IAESHelper aesHelper;

        /// <summary>
        /// Initializes a new instance of the <see cref="PasswordController"/> class.
        /// </summary>
        /// <param name="accountServices">
        /// The account services.
        /// </param>
        /// <param name="aesHelper">
        /// The aes helper.
        /// </param>
        /// <param name="accountCredentialServices">
        /// The account Credential Services.
        /// </param>
        /// <param name="auditLogServices">
        /// The audit Log Services.
        /// </param>
        public PasswordController(IAccountService accountServices, IAESHelper aesHelper, IAccountCredentialService accountCredentialServices, IAuditLogService auditLogServices)
        {
            this.accountServices = accountServices;
            this.aesHelper = aesHelper;
            this.accountCredentialServices = accountCredentialServices;
            this.auditLogServices = auditLogServices;
        }

        #region Create

        /// <summary>
        /// The create.
        /// </summary>
        /// <param name="id">
        /// The id.
        /// </param>
        /// <returns>
        /// The <see cref="IActionResult"/>.
        /// </returns>
        public async Task<IActionResult> Create(string id)
        {
            try
            {
                if (string.IsNullOrEmpty(id))
                {
                    return this.View($"_NotFound");
                }

                var decrypted = this.aesHelper.Decode(id);
                var accountId = Convert.ToInt32(decrypted);
                var account = await this.accountServices.FindAsync(accountId);
                if (account == null || account.AccountId == 0)
                {
                    return this.View($"_NotFound");
                }

                return this.View(account);
            }
            catch
            {
                return this.View($"_Error");
            }
        }

        /// <summary>
        /// The Create.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpPost]
        [AllowAnonymous]
        public async Task<IActionResult> Create([FromBody]ChangePasswordRequest model) => this.Json(await this.UpdatePasswordAsync(model));

        #endregion

        #region Reset

        /// <summary>
        /// The reset.
        /// </summary>
        /// <param name="id">
        /// The id.
        /// </param>
        /// <returns>
        /// The <see cref="IActionResult"/>.
        /// </returns>
        public async Task<IActionResult> Reset(string id)
        {
            try
            {
                if (string.IsNullOrEmpty(id))
                {
                    return this.View("_NotFound");
                }

                var decrypted = this.aesHelper.Decode(id);
                var resetDate = DateTimeFilter.ToDateTime(decrypted.Split(".")[1]);
                if (resetDate == null)
                {
                    return this.View("_Expired");
                }

                if (DateTimeFilter.Difference(Convert.ToDateTime(resetDate), DateTime.Now, CalendarType.Hours) >= 1)
                {
                    return this.View("_Expired");
                }

                var accountId = Convert.ToInt32(decrypted.Split(".")[0]);
                var account = await this.accountServices.FindAsync(accountId);
                if (account == null || account.AccountId == 0)
                {
                    return this.View("_NotFound");
                }

                return this.View(account);
            }
            catch
            {
                return this.View("_Error");
            }
        }

        /// <summary>
        /// The Reset.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpPost]
        [AllowAnonymous]
        public async Task<IActionResult> Reset([FromBody]ChangePasswordRequest model) => this.Json(await this.UpdatePasswordAsync(model));

        #endregion

        /// <summary>
        /// The update password async.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [NonAction]
        private async Task<bool> UpdatePasswordAsync(ChangePasswordRequest model)
        {
            var created = await this.accountCredentialServices.CreateAsync(model.AccountId, model.Password);
            if (created <= 0)
            {
                return false;
            }

            var account = await this.accountServices.FindAsync(model.AccountId);
            var auditLogModel = new AuditLogModel
            {
                AccountId = account.AccountId,
                LogTypeId = (int)LogTypes.Accounts,
                LogFrom = (short)account.RoleId,
                LogDate = DateTime.UtcNow,
                LogDescription = $"{account.FullName} ({account.RoleName}) account's password has been updated successfully."
            };
            await this.auditLogServices.LogAsync(auditLogModel);

            return true;
        }
    }
}